namespace Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Tests.Controllers
{
    using System.Collections.Generic;
    using System.Linq;
    using System.Net.Mail;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Controllers;
    using Microsoft.Samples.PlanMyNight.AddIns.EmailItinerary.Notifications;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;
    using MvcContrib.TestHelper;
    using Rhino.Mocks;

    [TestClass]
    public class EmailItineraryControllerFixture
    {
        [TestMethod]
        public void ImportingConstructorCallServiceFactory()
        {
            var serviceFactory = new Mock<IServiceFactory>();
            serviceFactory.Setup(o => o.GetItineraryContainerInstance()).Returns(new Mock<IItineraryContainer>().Object);
            serviceFactory.Setup(o => o.GetItinerariesRepositoryInstance()).Returns(new Mock<IItinerariesRepository>().Object);
            serviceFactory.Setup(o => o.GetActivitiesRepositoryInstance()).Returns(new Mock<IActivitiesRepository>().Object);

            new EmailItineraryController(serviceFactory.Object);
            serviceFactory.VerifyAll();
        }

        [TestMethod]
        public void EmailCurrentItineraryRedirectsToSuccess()
        {
            var itinerary = new Itinerary() { Name = "Night Out" };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var notifier = new Mock<INotificationService>();
            notifier.Setup(o => o.SendMail(It.IsAny<string>(), It.IsAny<MailAddress>(), It.IsAny<MailAddress>(), It.IsAny<Dictionary<string, string>>()))
                    .Returns(true);

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, activitiesRepository.Object, notifier.Object);
            builder.InitializeController(controller);

            var result = controller.Email("user@test.com", null);
            result.AssertResultIs<RedirectToRouteResult>();
            notifier.Verify(o => o.SendMail("ItineraryTemplate", It.Is<MailAddress>(a => a.Address == "test@sender.com"), It.Is<MailAddress>(a => a.Address == "user@test.com"), It.IsAny<Dictionary<string, string>>()));
            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void EmailCurrentItineraryUsingAjaxReturnsJsonResult()
        {
            var itinerary = new Itinerary() { Name = "Night Out" };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var notifier = new Mock<INotificationService>();
            notifier.Setup(o => o.SendMail(It.IsAny<string>(), It.IsAny<MailAddress>(), It.IsAny<MailAddress>(), It.IsAny<Dictionary<string, string>>()))
                    .Returns(true);

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Stub(r => r.ContentType).Return("application/json");
            var controller = new EmailItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, activitiesRepository.Object, notifier.Object);
            builder.InitializeController(controller);

            var result = controller.Email("user@test.com", null);
            result.AssertResultIs<JsonResult>();
            notifier.Verify(o => o.SendMail("ItineraryTemplate", It.Is<MailAddress>(a => a.Address == "test@sender.com"), It.Is<MailAddress>(a => a.Address == "user@test.com"), It.IsAny<Dictionary<string, string>>()));
            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void EmailSpecificItineraryRedirectsToSuccess()
        {
            var itinerary = new Itinerary() { Name = "Night Out", Id = 15, IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.Retrieve(15)).Returns(itinerary);

            var activitiesRepository = new Mock<IActivitiesRepository>();
            activitiesRepository.Setup(o => o.PopulateItineraryActivities(It.IsAny<Itinerary>()))
                                .Callback((Itinerary i) =>
                                {
                                    i.Activities.Where(a => a.ActivityId == "1").SingleOrDefault().Activity = new Activity { Id = "1", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Activity = new Activity { Id = "6", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                    i.Activities.Where(a => a.ActivityId == "10").SingleOrDefault().Activity = new Activity { Id = "10", Name = "RestoBar", PhoneNumber = "555-5555", City = "New York", State = "NY", Street = "Address 123", Zip = "1234" };
                                });

            var notifier = new Mock<INotificationService>();
            notifier.Setup(o => o.SendMail(It.IsAny<string>(), It.IsAny<MailAddress>(), It.IsAny<MailAddress>(), It.IsAny<Dictionary<string, string>>()))
                    .Returns(true);

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(new Mock<IItineraryContainer>().Object, repository.Object, activitiesRepository.Object, notifier.Object);
            builder.InitializeController(controller);

            var result = controller.Email("user@test.com", 15);
            result.AssertResultIs<RedirectToRouteResult>();
            notifier.Verify(o => o.SendMail("ItineraryTemplate", It.Is<MailAddress>(a => a.Address == "test@sender.com"), It.Is<MailAddress>(a => a.Address == "user@test.com"), It.IsAny<Dictionary<string, string>>()));
            activitiesRepository.VerifyAll();
        }

        [TestMethod]
        public void EmailSpecificItineraryWhichIsNotPublicAndNotMineReturnsEmptyResult()
        {
            var itinerary = new Itinerary() { Name = "Night Out", Id = 15, IsPublic = false };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var repository = new Mock<IItinerariesRepository>();
            repository.Setup(o => o.Retrieve(15)).Returns(itinerary);

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(new Mock<IItineraryContainer>().Object, repository.Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var result = controller.Email("user@test.com", 15);
            result.AssertResultIs<EmptyResult>();
            repository.VerifyAll();
        }

        [TestMethod]
        public void EmailCurrentEmptyOrNullItineraryReturnsEmptyResult()
        {
            var itinerary = new Itinerary();

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var notifier = new Mock<INotificationService>();

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, notifier.Object);
            builder.InitializeController(controller);

            var result = controller.Email("user@test.com", null);
            result.AssertResultIs<EmptyResult>();
        }

        [TestMethod]
        public void EmailSuccess()
        {
            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(new Mock<IItineraryContainer>().Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var result = controller.EmailSuccess();
            result.AssertViewRendered();
        }

        [TestMethod]
        public void GetEmailToolboxShouldReturnPartialViewWithLink()
        {
            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(new Mock<IItineraryContainer>().Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var result = controller.GetEmailToolbox(1);
            var view = result.AssertResultIs<PartialViewResult>();
            StringAssert.Contains(view.ViewName, "EmailToolbox");
            Assert.IsNotNull(view.ViewData["id"]);
        }

        [TestMethod]
        public void GetEmailToolboxWithIdShouldReturnPartialViewWithLinkToId()
        {
            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(new Mock<IItineraryContainer>().Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var result = controller.GetEmailToolbox(1);
            var view = result.AssertResultIs<PartialViewResult>();
            StringAssert.Contains(view.ViewName, "EmailToolbox");
            Assert.AreEqual(1L, view.ViewData["id"]);
        }

        [TestMethod]
        public void EmailReturnsEmailView()
        {
            var itinerary = new Itinerary() { Name = "Night Out" };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var viewResult = controller.Email(null).AssertViewRendered();
            Assert.AreEqual("Email", viewResult.ViewName);
            Assert.IsFalse(viewResult.ViewData.ContainsKey("id"));
        }

        [TestMethod]
        public void EmailWithItineraryIdReturnsEmailViewAndViewDataWithId()
        {
            var itinerary = new Itinerary() { Name = "Night Out" };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var builder = new TestControllerBuilder();
            var controller = new EmailItineraryController(container.Object, new Mock<IItinerariesRepository>().Object, new Mock<IActivitiesRepository>().Object, new Mock<INotificationService>().Object);
            builder.InitializeController(controller);

            var viewResult = controller.Email(10).AssertViewRendered();
            Assert.AreEqual("Email", viewResult.ViewName);
            Assert.AreEqual(10L, viewResult.ViewData["id"]);
        }
    }
}
